# This R code fits the various models to the case-study datasets `ovenCHp' and `OVpossumCH' 
# using the R package `openCR' 2.1.0 that may be found at https://CRAN.R-project.org/package=openCR.

#------------------------------------------------------------------------------

library(openCR)
setNumThreads(7)  # customize to match the number of cores available

#------------------------------------------------------------------------------
# Code used for both datasets
# Construct data frame of models to fit, one per row
scen <- expand.grid(
    movementmodel = c('static','BVN','BVE','BVT', 'BVNzi','BVEzi'), 
    sparse        = c(FALSE,TRUE), 
    stringsAsFactors = FALSE
)
# Function to fit one scenario and save result to a file
fitone <- function (movementmodel, sparse, filestem, ...) {
    fit <- openCR.fit(movementmodel = movementmodel, sparsekernel = sparse, ...)
    filename <- paste0(filestem, movementmodel, sparse, '.RDS')
    saveRDS(fit, file = filename)
}
# Read output files and combine as list of openCR objects
combineRDS <- function (filestem, scen) {
    sparselist <- lapply(1:nrow(scen), 
        function (s) {
            movementmodel <- scen$movementmodel[s]
            sparse <- scen$sparse[s]
            filename <- paste0(filestem, movementmodel, sparse, '.RDS')
            if (file.exists(filename)) readRDS(filename) else NULL
        }
    )
    ok <- sapply(sparselist, inherits, 'openCR')
    sparselist <- openCRlist(sparselist[ok])
    names(sparselist) <- paste(scen$movementmodel[ok], scen$sparse[ok], sep='.')
    sparselist    
}
#------------------------------------------------------------------------------

# a. Ovenbird
ovenmask <- make.mask(traps(ovenCHp[[1]]), buffer = 500, spacing = 20,
    type = 'trapbuffer')
for (s in 1:nrow(scen)) {
    fitone(
        movementmodel = scen$movementmodel[s], 
        sparse        = scen$sparse[s],
        filestem      = 'oven',
        type          = 'PLBsecrf',
        capthist      = ovenCHp, 
        mask          = ovenmask,
        kernelradius  = 30
    )
}
ovensparse <- combineRDS('oven', scen)
AIC(ovensparse)  # etc.
#------------------------------------------------------------------------------

# b. Brushtail possum

# Prepare mask excluding shingle riverbed
library(rgdal)
datadir <- system.file("extdata", package = "secr")
OVforest <- rgdal::readOGR(dsn = datadir, layer = "OVforest")
OVmask <- make.mask(traps(OVpossumCH[[1]]), buffer = 150, spacing = 10,
    type = 'trapbuffer', poly = OVforest[1:2,])

# Fit model with season-specific survival and recruitment
for (s in 1:nrow(scen)) {
    fitone(
        movementmodel = scen$movementmodel[s], 
        sparse        = scen$sparse[s],
        filestem      = 'OV',
        type          = 'PLBsecrf',
        capthist      = OVpossumCH, 
        mask          = OVmask,
        kernelradius  = 30,
        model         = list(phi~season, f~season),
        sessioncov    = data.frame(season=factor(c(1,2,3,1,2,3)))
    )
}
OVsparse <- combineRDS('OV', scen)
AIC(OVsparse)  # etc.
#------------------------------------------------------------------------------

